/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.timeseries.caching;

import com.amazon.randomcutforest.parkservices.ThresholdedRandomCutForest;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.timeseries.ExpiringState;
import org.opensearch.timeseries.MemoryTracker;
import org.opensearch.timeseries.caching.PriorityTracker;
import org.opensearch.timeseries.indices.IndexManagement;
import org.opensearch.timeseries.ml.CheckpointDao;
import org.opensearch.timeseries.ml.ModelState;
import org.opensearch.timeseries.ratelimit.CheckpointMaintainRequest;
import org.opensearch.timeseries.ratelimit.CheckpointMaintainWorker;
import org.opensearch.timeseries.ratelimit.CheckpointWriteWorker;
import org.opensearch.timeseries.ratelimit.RateLimitedRequestWorker;
import org.opensearch.timeseries.ratelimit.RequestPriority;
import org.opensearch.timeseries.util.DateUtils;

public abstract class CacheBuffer<RCFModelType extends ThresholdedRandomCutForest, IndexType extends Enum<IndexType>, IndexManagementType extends IndexManagement<IndexType>, CheckpointDaoType extends CheckpointDao<RCFModelType, IndexType, IndexManagementType>, CheckpointWriterType extends CheckpointWriteWorker<RCFModelType, IndexType, IndexManagementType, CheckpointDaoType>, CheckpointMaintainerType extends CheckpointMaintainWorker>
implements ExpiringState {
    private static final Logger LOG = LogManager.getLogger(CacheBuffer.class);
    protected Instant lastUsedTime;
    protected final Clock clock;
    protected final MemoryTracker memoryTracker;
    protected int checkpointIntervalHrs;
    protected final Duration modelTtl;
    protected int minimumCapacity;
    protected final long memoryConsumptionPerModel;
    protected long reservedBytes;
    protected final CheckpointWriterType checkpointWriteQueue;
    protected final CheckpointMaintainerType checkpointMaintainQueue;
    protected final String configId;
    protected final MemoryTracker.Origin origin;
    protected final PriorityTracker priorityTracker;
    protected final ConcurrentHashMap<String, ModelState<RCFModelType>> items;

    public CacheBuffer(int minimumCapacity, Clock clock, MemoryTracker memoryTracker, int checkpointIntervalHrs, Duration modelTtl, long memoryConsumptionPerEntity, CheckpointWriterType checkpointWriteQueue, CheckpointMaintainerType checkpointMaintainQueue, String configId, MemoryTracker.Origin origin, PriorityTracker priorityTracker) {
        this.lastUsedTime = clock.instant();
        this.clock = clock;
        this.memoryTracker = memoryTracker;
        this.setCheckpointIntervalHrs(checkpointIntervalHrs);
        this.modelTtl = modelTtl;
        this.memoryConsumptionPerModel = memoryConsumptionPerEntity;
        this.checkpointWriteQueue = checkpointWriteQueue;
        this.checkpointMaintainQueue = checkpointMaintainQueue;
        this.configId = configId;
        this.origin = origin;
        this.priorityTracker = priorityTracker;
        this.items = new ConcurrentHashMap();
        this.setMinimumCapacity(minimumCapacity);
    }

    public void setMinimumCapacity(int minimumCapacity) {
        if (minimumCapacity < 0) {
            throw new IllegalArgumentException("minimum capacity should be larger than or equal 0");
        }
        this.minimumCapacity = minimumCapacity;
        this.reservedBytes = this.memoryConsumptionPerModel * (long)minimumCapacity;
    }

    @Override
    public boolean expired(Duration stateTtl) {
        return this.expired(this.lastUsedTime, stateTtl, this.clock.instant());
    }

    public void setCheckpointIntervalHrs(int checkpointIntervalHrs) {
        this.checkpointIntervalHrs = checkpointIntervalHrs;
        if (checkpointIntervalHrs <= 0) {
            this.checkpointIntervalHrs = 1;
        }
    }

    public int getCheckpointIntervalHrs() {
        return this.checkpointIntervalHrs;
    }

    public long getReservedBytes() {
        return this.reservedBytes;
    }

    public long getMemoryConsumptionPerModel() {
        return this.memoryConsumptionPerModel;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        if (obj instanceof CacheBuffer) {
            CacheBuffer other = (CacheBuffer)obj;
            EqualsBuilder equalsBuilder = new EqualsBuilder();
            equalsBuilder.append((Object)this.configId, (Object)other.configId);
            return equalsBuilder.isEquals();
        }
        return false;
    }

    public int hashCode() {
        return new HashCodeBuilder().append((Object)this.configId).toHashCode();
    }

    public String getConfigId() {
        return this.configId;
    }

    public boolean sharedCacheEmpty() {
        return this.items.size() <= this.minimumCapacity;
    }

    public long getBytesInSharedCache() {
        int sharedCacheEntries = this.items.size() - this.minimumCapacity;
        if (sharedCacheEntries > 0) {
            return this.memoryConsumptionPerModel * (long)sharedCacheEntries;
        }
        return 0L;
    }

    public void clear() {
        this.memoryTracker.releaseMemory(this.getReservedBytes(), true, this.origin);
        if (!this.sharedCacheEmpty()) {
            this.memoryTracker.releaseMemory(this.getBytesInSharedCache(), false, this.origin);
        }
        this.items.clear();
        this.priorityTracker.clearPriority();
    }

    public List<ModelState<RCFModelType>> maintenance() {
        ArrayList modelsToSave = new ArrayList();
        ArrayList<ModelState<RCFModelType>> removedStates = new ArrayList<ModelState<RCFModelType>>();
        Instant now = this.clock.instant();
        int currentHour = DateUtils.getUTCHourOfDay(now);
        int currentSlot = currentHour % this.checkpointIntervalHrs;
        this.items.entrySet().stream().forEach(entry -> {
            String entityModelId = (String)entry.getKey();
            try {
                ModelState modelState = (ModelState)entry.getValue();
                if (modelState.getLastUsedTime().plus(this.modelTtl).isBefore(now)) {
                    removedStates.add(this.remove(entityModelId));
                } else if (Math.abs(entityModelId.hashCode()) % this.checkpointIntervalHrs == currentSlot) {
                    modelsToSave.add(new CheckpointMaintainRequest(System.currentTimeMillis() + this.modelTtl.toMillis(), this.configId, RequestPriority.LOW, entityModelId));
                }
            }
            catch (Exception e) {
                LOG.warn("Failed to finish maintenance for model id " + entityModelId, (Throwable)e);
            }
        });
        ((RateLimitedRequestWorker)this.checkpointMaintainQueue).putAll(modelsToSave);
        return removedStates;
    }

    public ModelState<RCFModelType> remove(String keyToRemove, boolean saveCheckpoint) {
        this.priorityTracker.removePriority(keyToRemove);
        boolean reserved = this.sharedCacheEmpty();
        ModelState<RCFModelType> valueRemoved = this.items.remove(keyToRemove);
        if (valueRemoved != null) {
            if (!reserved) {
                this.memoryTracker.releaseMemory(this.memoryConsumptionPerModel, false, this.origin);
            }
            if (saveCheckpoint) {
                ((CheckpointWriteWorker)this.checkpointWriteQueue).write(valueRemoved, valueRemoved.getModel().isEmpty(), RequestPriority.MEDIUM);
            }
            valueRemoved.clear();
        }
        return valueRemoved;
    }

    public ModelState<RCFModelType> remove(String keyToRemove) {
        return this.remove(keyToRemove, true);
    }

    public PriorityTracker getPriorityTracker() {
        return this.priorityTracker;
    }

    public ModelState<RCFModelType> remove() {
        Optional<String> key = this.priorityTracker.getMinimumPriorityEntityId();
        if (key.isPresent()) {
            return this.remove(key.get());
        }
        return null;
    }

    public boolean canRemove() {
        return !this.items.isEmpty() && this.items.size() > this.minimumCapacity;
    }

    public boolean dedicatedCacheAvailable() {
        return this.items.size() < this.minimumCapacity;
    }

    public int getActiveEntities() {
        return this.items.size();
    }

    public boolean isActive(String entityModelId) {
        return this.items.containsKey(entityModelId);
    }

    public long getLastUsedTime(String entityModelId) {
        ModelState<RCFModelType> state = this.items.get(entityModelId);
        if (state != null) {
            return state.getLastUsedTime().toEpochMilli();
        }
        return -1L;
    }

    public ModelState<RCFModelType> getModelState(String entityModelId) {
        return this.items.get(entityModelId);
    }

    private void update(String entityModelId) {
        this.priorityTracker.updatePriority(entityModelId);
        Instant now = this.clock.instant();
        this.items.get(entityModelId).setLastUsedTime(now);
        this.lastUsedTime = now;
    }

    public void put(String entityModelId, ModelState<RCFModelType> value) {
        this.put(entityModelId, value, value.getPriority());
    }

    private void put(String entityModelId, ModelState<RCFModelType> value, float priority) {
        ModelState<RCFModelType> contentNode = this.items.get(entityModelId);
        if (contentNode == null) {
            this.priorityTracker.addPriority(entityModelId, priority);
            this.items.put(entityModelId, value);
            Instant now = this.clock.instant();
            value.setLastUsedTime(now);
            this.lastUsedTime = now;
            if (!this.sharedCacheEmpty()) {
                this.memoryTracker.consumeMemory(this.memoryConsumptionPerModel, false, this.origin);
            }
        } else {
            this.update(entityModelId);
            this.items.put(entityModelId, value);
        }
    }

    public ModelState<RCFModelType> get(String key) {
        ModelState<RCFModelType> node = this.items.get(key);
        if (node == null) {
            return null;
        }
        this.update(key);
        return node;
    }

    public ModelState<RCFModelType> getWithoutUpdatePriority(String key) {
        ModelState<RCFModelType> node = this.items.get(key);
        if (node == null) {
            return null;
        }
        return node;
    }

    public boolean canReplaceWithinConfig(float priority) {
        if (this.items.isEmpty()) {
            return false;
        }
        Optional<Map.Entry<String, Float>> minPriorityItem = this.priorityTracker.getMinimumPriority();
        return minPriorityItem.isPresent() && priority > minPriorityItem.get().getValue().floatValue();
    }

    public ModelState<RCFModelType> replace(String entityModelId, ModelState<RCFModelType> value) {
        ModelState<RCFModelType> replaced = this.remove();
        this.put(entityModelId, value);
        return replaced;
    }

    public List<ModelState<RCFModelType>> getAllModelStates() {
        return this.items.values().stream().collect(Collectors.toList());
    }
}

