"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPPLLanguageAnalyzer = exports.PPLLanguageAnalyzer = void 0;
const tslib_1 = require("tslib");
const antlr = tslib_1.__importStar(require("antlr4ng"));
const antlr_grammar_1 = require("@osd/antlr-grammar");
const ppl_error_listener_1 = require("./ppl_error_listener");
/**
 * PPL Language Analyzer - provides tokenization, validation, and code completion for PPL queries
 * Uses ANTLR generated lexer and parser for accurate language processing
 */
class PPLLanguageAnalyzer {
    constructor() {
        // ANTLR-based language analyzer initialization
    }
    /**
     * Creates and configures ANTLR lexer and token stream from input code
     */
    createLexerAndTokenStream(code) {
        const inputStream = antlr.CharStream.fromString(code);
        const lexer = new antlr_grammar_1.SimplifiedOpenSearchPPLLexer(inputStream);
        const tokenStream = new antlr.CommonTokenStream(lexer);
        return { lexer, tokenStream };
    }
    /**
     * Creates and configures ANTLR parser with error listeners
     */
    createParserWithErrorHandling(tokenStream) {
        const parser = new antlr_grammar_1.SimplifiedOpenSearchPPLParser(tokenStream);
        // Set up error listeners
        const lexerErrorListener = new ppl_error_listener_1.PPLSyntaxErrorListener();
        const parserErrorListener = new ppl_error_listener_1.PPLSyntaxErrorListener();
        parser.removeErrorListeners();
        parser.addErrorListener(parserErrorListener);
        return { parser, lexerErrorListener, parserErrorListener };
    }
    /**
     * Tokenize PPL code into tokens using ANTLR lexer
     */
    tokenize(code) {
        const tokens = [];
        try {
            const { lexer, tokenStream } = this.createLexerAndTokenStream(code);
            tokenStream.fill();
            const antlrTokens = tokenStream.getTokens();
            for (const token of antlrTokens) {
                if (token.type !== antlr.Token.EOF) {
                    const tokenTypeName = this.getTokenTypeName(token.type, lexer);
                    tokens.push({
                        type: tokenTypeName,
                        value: token.text || '',
                        startIndex: token.start,
                        stopIndex: token.stop,
                        line: token.line,
                        column: token.column,
                    });
                }
            }
        }
        catch (error) {
            // Silent error handling for tokenization issues
        }
        return tokens;
    }
    /**
     * Validate PPL code using ANTLR parser
     */
    validate(code) {
        try {
            const { lexer, tokenStream } = this.createLexerAndTokenStream(code);
            // Add error listener to lexer
            const lexerErrorListener = new ppl_error_listener_1.PPLSyntaxErrorListener();
            lexer.removeErrorListeners();
            lexer.addErrorListener(lexerErrorListener);
            const { parser, parserErrorListener } = this.createParserWithErrorHandling(tokenStream);
            parser.root();
            // Collect all errors from both lexer and parser
            const allErrors = [...lexerErrorListener.errors, ...parserErrorListener.errors];
            if (allErrors.length > 0) {
                return {
                    isValid: false,
                    errors: allErrors,
                };
            }
            return {
                isValid: true,
                errors: [],
            };
        }
        catch (error) {
            // Return parsing exception as error
            return {
                isValid: false,
                errors: [
                    {
                        message: error instanceof Error ? error.message : String(error),
                        line: 1,
                        column: 0,
                        endLine: 1,
                        endColumn: 1,
                    },
                ],
            };
        }
    }
    /**
     * Get token type name from ANTLR token type
     */
    getTokenTypeName(tokenType, lexer) {
        const vocabulary = lexer.vocabulary;
        const symbolicName = vocabulary.getSymbolicName(tokenType);
        if (symbolicName) {
            return symbolicName.toLowerCase();
        }
        const literalName = vocabulary.getLiteralName(tokenType);
        if (literalName) {
            return literalName.replace(/['"]/g, '');
        }
        return 'unknown';
    }
}
exports.PPLLanguageAnalyzer = PPLLanguageAnalyzer;
/**
 * Singleton instance of PPL Language Analyzer
 * Provides a shared instance for efficient memory usage across the application
 */
let pplLanguageAnalyzerInstance = null;
/**
 * Get or create the singleton instance of PPL Language Analyzer
 */
const getPPLLanguageAnalyzer = () => {
    if (!pplLanguageAnalyzerInstance) {
        pplLanguageAnalyzerInstance = new PPLLanguageAnalyzer();
    }
    return pplLanguageAnalyzerInstance;
};
exports.getPPLLanguageAnalyzer = getPPLLanguageAnalyzer;
//# sourceMappingURL=ppl_language_analyzer.js.map