/*
 * Decompiled with CFR 0.152.
 */
package ghidra.formats.gfilesystem;

import docking.widgets.OptionDialog;
import generic.hash.HashUtilities;
import ghidra.app.util.bin.ByteProvider;
import ghidra.formats.gfilesystem.FSRL;
import ghidra.formats.gfilesystem.FSRLRoot;
import ghidra.formats.gfilesystem.GFile;
import ghidra.formats.gfilesystem.GFileSystem;
import ghidra.formats.gfilesystem.annotations.FileSystemInfo;
import ghidra.formats.gfilesystem.fileinfo.FileType;
import ghidra.util.Msg;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.CryptoException;
import ghidra.util.task.TaskMonitor;
import java.awt.Component;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;
import org.apache.commons.io.FilenameUtils;
import util.CollectionUtils;
import utilities.util.FileUtilities;

public class FSUtilities {
    public static final String SEPARATOR_CHARS = "/\\:";
    public static final String SEPARATOR = "/";
    private static final char DOT = '.';
    private static final TimeZone GMT = TimeZone.getTimeZone("GMT");
    private static char[] hexdigit = "0123456789abcdef".toCharArray();
    public static final Comparator<GFile> GFILE_NAME_TYPE_COMPARATOR = (o1, o2) -> {
        int result = Boolean.compare(!o1.isDirectory(), !o2.isDirectory());
        if (result == 0) {
            String n1 = Objects.requireNonNullElse(o1.getName(), "");
            String n2 = Objects.requireNonNullElse(o2.getName(), "");
            result = n1.compareToIgnoreCase(n2);
        }
        return result;
    };

    public static String infoMapToString(Map<String, String> info) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : info.entrySet()) {
            sb.append(entry.getKey()).append(": ").append(entry.getValue()).append("\n");
        }
        return sb.toString();
    }

    public static String getSafeFilename(String untrustedFilename) {
        switch (untrustedFilename = untrustedFilename.replaceAll("[/\\\\:|]", "_").trim()) {
            case "": {
                return "empty_filename";
            }
            case ".": {
                return "dot";
            }
            case "..": {
                return "dotdot";
            }
        }
        return FSUtilities.escapeEncode(untrustedFilename);
    }

    public static String escapeEncode(String s) {
        if (s == null) {
            return null;
        }
        String escapeChars = "%?|";
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (c < ' ' || c > '~' || escapeChars.indexOf(c) >= 0) {
                FSUtilities.appendHexEncoded(sb, c);
                continue;
            }
            sb.append(c);
        }
        return sb.toString();
    }

    public static String escapeDecode(String s) throws MalformedURLException {
        if (s == null) {
            return null;
        }
        byte[] bytes = null;
        StringBuilder sb = new StringBuilder();
        int i = 0;
        while (i < s.length()) {
            char c = s.charAt(i);
            if (c == '%') {
                if (bytes == null) {
                    bytes = new byte[(s.length() - i) / 3];
                }
                int pos = 0;
                while (i + 2 < s.length() && c == '%') {
                    int v = Integer.parseInt(s.substring(i + 1, i + 3), 16);
                    if (v < 0) {
                        throw new MalformedURLException("Bad hex characters in escape (%) pattern: " + s);
                    }
                    bytes[pos++] = (byte)v;
                    if ((i += 3) >= s.length()) continue;
                    c = s.charAt(i);
                }
                if (i < s.length() && c == '%') {
                    throw new MalformedURLException("Bad escape pattern in " + s);
                }
                sb.append(new String(bytes, 0, pos, StandardCharsets.UTF_8));
                continue;
            }
            sb.append(c);
            ++i;
        }
        return sb.toString();
    }

    private static void appendHexEncoded(StringBuilder sb, char c) {
        if (c < '\u0080') {
            sb.append('%').append(hexdigit[c >> 4]).append(hexdigit[c & 0xF]);
            return;
        }
        sb.append(URLEncoder.encode("" + c, StandardCharsets.UTF_8));
    }

    @Deprecated(forRemoval=true, since="11.4")
    public static List<GFile> listFileSystem(GFileSystem fs, GFile dir, List<GFile> result, TaskMonitor taskMonitor) throws IOException, CancelledException {
        if (result == null) {
            result = new ArrayList<GFile>();
        }
        for (GFile gFile : fs.getListing(dir)) {
            taskMonitor.checkCancelled();
            if (gFile.isDirectory()) {
                FSUtilities.listFileSystem(fs, gFile, result, taskMonitor);
                continue;
            }
            result.add(gFile);
        }
        return result;
    }

    public static String getFilesystemTypeFromClass(Class<?> clazz) {
        FileSystemInfo fsi = clazz.getAnnotation(FileSystemInfo.class);
        return fsi != null ? fsi.type() : null;
    }

    public static String getFilesystemDescriptionFromClass(Class<?> clazz) {
        FileSystemInfo fsi = clazz.getAnnotation(FileSystemInfo.class);
        return fsi != null ? fsi.description() : null;
    }

    public static int getFilesystemPriorityFromClass(Class<?> clazz) {
        FileSystemInfo fsi = clazz.getAnnotation(FileSystemInfo.class);
        return fsi != null ? fsi.priority() : 0;
    }

    public static boolean isSameFS(List<FSRL> fsrls) {
        if (fsrls.isEmpty()) {
            return true;
        }
        FSRLRoot fsFSRL = fsrls.get(0).getFS();
        for (FSRL fsrl : fsrls) {
            if (fsFSRL.equals(fsrl.getFS())) continue;
            return false;
        }
        return true;
    }

    public static void displayException(Object originator, Component parent, String title, String message, Throwable throwable) {
        if (throwable instanceof CryptoException) {
            FSUtilities.displayCryptoException(originator, parent, title, message, (CryptoException)throwable);
        } else {
            Msg.showError((Object)originator, (Component)parent, (String)title, (Object)message, (Throwable)throwable);
        }
    }

    private static void displayCryptoException(Object originator, Component parent, String title, String message, CryptoException ce) {
        String ce_msg = ce.getMessage();
        if (ce_msg.contains("Install the JCE")) {
            File javaHomeDir = new File(System.getProperty("java.home"));
            File libSecurityDir = new File(new File(javaHomeDir, "lib"), "security");
            if (OptionDialog.showYesNoDialog((Component)parent, (String)title, (String)("<html><div style='margin-bottom: 20pt'>A problem with the Java crypto subsystem was encountered:</div><div style='font-weight: bold; margin-bottom: 20pt; margin-left: 50pt'>" + ce_msg + "</div><div style='margin-bottom: 20pt'>Which caused:</div><div style='font-weight: bold; margin-bottom: 20pt; margin-left: 50pt'>" + message + "</div><div style='margin-bottom: 20pt'>This may be fixed by installing the unlimited strength JCE into your JRE's \"lib/security\" directory.</div><div style='margin-bottom: 20pt'>The unlimited strength JCE should be available from the same download location as your JRE.</div><div>Display your JRE's \"lib/security\" directory?</div></html>")) == 1) {
                try {
                    FileUtilities.openNative((File)libSecurityDir);
                }
                catch (IOException e) {
                    Msg.showError((Object)originator, (Component)parent, (String)"Problem starting explorer", (Object)("Problem starting file explorer: " + e.getMessage()));
                }
            }
            return;
        }
        Msg.showWarn((Object)originator, (Component)parent, (String)title, (Object)(message + ": " + ce_msg));
    }

    public static long copyByteProviderToFile(ByteProvider provider, File destFile, TaskMonitor monitor) throws IOException, CancelledException {
        try (InputStream is = provider.getInputStream(0L);){
            long l;
            try (FileOutputStream fos = new FileOutputStream(destFile);){
                l = FSUtilities.streamCopy(is, fos, monitor);
            }
            return l;
        }
    }

    public static long streamCopy(InputStream is, OutputStream os, TaskMonitor monitor) throws IOException, CancelledException {
        int bytesRead;
        byte[] buffer = new byte[32768];
        long totalBytesCopied = 0L;
        while ((bytesRead = is.read(buffer)) > 0) {
            os.write(buffer, 0, bytesRead);
            monitor.setProgress(totalBytesCopied += (long)bytesRead);
            monitor.checkCancelled();
        }
        os.flush();
        return totalBytesCopied;
    }

    public static List<String> getLines(ByteProvider byteProvider) throws IOException {
        try (InputStream is = byteProvider.getInputStream(0L);){
            List list = FileUtilities.getLines((InputStream)is);
            return list;
        }
    }

    public static String getFileMD5(File f, TaskMonitor monitor) throws IOException, CancelledException {
        try (FileInputStream fis = new FileInputStream(f);){
            String string = FSUtilities.getMD5(fis, f.getName(), f.length(), monitor);
            return string;
        }
    }

    public static String getMD5(ByteProvider provider, TaskMonitor monitor) throws IOException, CancelledException {
        try (InputStream is = provider.getInputStream(0L);){
            String string = FSUtilities.getMD5(is, provider.getName(), provider.length(), monitor);
            return string;
        }
    }

    public static String getMD5(InputStream is, String name, long expectedLength, TaskMonitor monitor) throws IOException, CancelledException {
        try {
            int bytesRead;
            long startms;
            long prevElapsed = startms = System.currentTimeMillis();
            monitor.initialize(expectedLength, "Hashing %s".formatted(name));
            MessageDigest messageDigest = MessageDigest.getInstance(HashUtilities.MD5_ALGORITHM);
            int bufSize = (int)Math.max(1024L, Math.min(expectedLength, 0x100000L));
            byte[] buf = new byte[bufSize];
            long totalBytesRead = 0L;
            while ((bytesRead = is.read(buf)) >= 0) {
                messageDigest.update(buf, 0, bytesRead);
                monitor.increment((long)bytesRead);
                long now = System.currentTimeMillis();
                if (now - prevElapsed <= 5000L || (totalBytesRead += (long)bytesRead) <= (long)bufSize) continue;
                prevElapsed = now;
                long elapsed = now - startms;
                long rate = (long)((float)totalBytesRead / ((float)elapsed / 1000.0f));
                monitor.setMessage("Hashing %s %s/s".formatted(name, FileUtilities.formatLength((long)rate)));
            }
            return NumericUtilities.convertBytesToString((byte[])messageDigest.digest());
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException(e);
        }
    }

    public static String appendPath(String ... paths) {
        if (CollectionUtils.isAllNull(Arrays.asList(paths))) {
            return null;
        }
        StringBuilder buffer = new StringBuilder();
        for (String path : paths) {
            boolean pathStartsWithSlash;
            if (path == null || path.isEmpty()) continue;
            boolean emptyBuffer = buffer.length() == 0;
            boolean bufferEndsWithSlash = !emptyBuffer && "/\\".indexOf(buffer.charAt(buffer.length() - 1)) != -1;
            boolean bl = pathStartsWithSlash = "/\\".indexOf(path.charAt(0)) != -1;
            if (!(bufferEndsWithSlash || pathStartsWithSlash || emptyBuffer)) {
                buffer.append(SEPARATOR);
            } else if (pathStartsWithSlash && bufferEndsWithSlash) {
                path = path.substring(1);
            }
            buffer.append(path);
        }
        return buffer.toString();
    }

    public static String getExtension(String path, int extLevel) {
        Objects.requireNonNull(path);
        if (extLevel < 1) {
            throw new IllegalArgumentException("Bad extention level: " + extLevel);
        }
        for (int i = path.length() - 1; i >= 0; --i) {
            char c = path.charAt(i);
            if (SEPARATOR_CHARS.indexOf(c) != -1) {
                return null;
            }
            if (c != '.' || --extLevel != 0) continue;
            return path.substring(i);
        }
        return null;
    }

    public static String normalizeNativePath(String path) {
        return FSUtilities.appendPath(SEPARATOR, FilenameUtils.separatorsToUnix((String)path));
    }

    public static String formatFSTimestamp(Date d) {
        if (d == null) {
            return "NA";
        }
        SimpleDateFormat df = new SimpleDateFormat("dd MMM yyyy HH:mm:ss z");
        df.setTimeZone(GMT);
        return df.format(d);
    }

    public static String formatSize(Long length) {
        return length != null ? String.format("%d (%s)", length, FileUtilities.formatLength((long)length)) : "NA";
    }

    public static void uncheckedClose(AutoCloseable c, String msg) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            Msg.warn(FSUtilities.class, (Object)Objects.requireNonNullElse(msg, "Problem closing object"), (Throwable)e);
        }
    }

    public static boolean isSymlink(File f) {
        try {
            return f != null && Files.isSymbolicLink(f.toPath());
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static String readSymlink(File f) {
        try {
            Path symlink = Files.readSymbolicLink(f.toPath());
            return symlink.toString();
        }
        catch (Throwable throwable) {
            return null;
        }
    }

    public static FileType getFileType(File f) {
        try {
            Path p = f.toPath();
            if (Files.isSymbolicLink(p)) {
                return FileType.SYMBOLIC_LINK;
            }
            if (Files.isDirectory(p, new LinkOption[0])) {
                return FileType.DIRECTORY;
            }
            if (Files.isRegularFile(p, new LinkOption[0])) {
                return FileType.FILE;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        return FileType.UNKNOWN;
    }

    public static String[] splitPath(String path) {
        return Objects.requireNonNullElse(path, "").replace('\\', '/').split(SEPARATOR);
    }

    public static String mirroredProjectPath(String path) {
        if (((String)(path = FSUtilities.normalizeNativePath((String)path))).length() >= 3 && ((String)path).charAt(0) == '/' && Character.isLetter(((String)path).charAt(1)) && ((String)path).charAt(2) == ':') {
            path = SEPARATOR + ((String)path).charAt(1) + ((String)path).substring(3);
        }
        return path;
    }
}

