# (c) Copyright 2021-2022. CodeWeavers, Inc.

from gi.repository import GLib
from gi.repository import Gtk

import bottlecollection
import bottlemanagement
import cxguitools
import cxutils
import installtask

from cxutils import cxgettext as _


class SelectBottleDialogController:

    def __init__(self, install_task, transient_for):
        self.install_task = install_task
        self.bottle_name = install_task.newBottleName
        self.target_bottle = install_task.target_bottle

        # Import widget modules so Gtk.Builder() finds them
        import cxhtmltextview # pylint: disable=W0612,W0611

        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('selectbottledialog'))
        self.xml.connect_signals(self)

        self.xml.get_object('SelectBottleDialog').set_transient_for(transient_for)

        self.xml.get_object('NewBottleNameEntry').set_text(self.bottle_name)
        self.update_new_bottle_widgets()

        self.load_icons()
        self.update_bottle_warnings()

        self.updating_bottles = False

        self.bottle_list_model = Gtk.ListStore(object, str, object, bool) # name, display markup, sort key, sensitive
        self.bottle_list_model.set_sort_func(2, self._bottle_list_cmp)
        self.bottle_list_model.set_sort_column_id(2, Gtk.SortType.ASCENDING)
        self.xml.get_object('BottleComboBox').set_model(self.bottle_list_model)

        self.template_list_model = Gtk.ListStore(object, str, object, bool) # name, display markup, sort key, sensitive
        self.template_list_model.set_sort_func(2, self._bottle_list_cmp)
        self.template_list_model.set_sort_column_id(2, Gtk.SortType.ASCENDING)
        self.xml.get_object('BottleTypeComboBox').set_model(self.template_list_model)

        self.update_bottles()

        GLib.idle_add(self.xml.get_object('NewBottleNameEntry').grab_focus)

    def run(self):
        return self.xml.get_object('SelectBottleDialog').run()

    def destroy(self):
        self.xml.get_object('SelectBottleDialog').destroy()

    def load_icons(self):
        warning_image = cxguitools.load_icon(('dialog-warning', 'gtk-dialog-warning'), Gtk.IconSize.MENU)
        if warning_image:
            self.xml.get_object('ErrorText').images['warning'] = warning_image

        error_image = cxguitools.load_icon(('dialog-error', 'gtk-dialog-error'), Gtk.IconSize.MENU)
        if error_image:
            self.xml.get_object('ErrorText').images['error'] = error_image

        check_image = cxguitools.load_icon(('dialog-apply', 'dialog-ok-apply', 'gtk-apply'), Gtk.IconSize.MENU)
        if check_image:
            self.xml.get_object('ErrorText').images['check'] = check_image

    def update_new_bottle_widgets(self):
        if self.target_bottle and self.target_bottle.is_template:
            self.xml.get_object('NewBottleWidgets').show()
        else:
            self.xml.get_object('NewBottleWidgets').hide()

    def update_bottle_warnings(self):
        if self.target_bottle and self.target_bottle.is_template and not self.install_task.is_new_bottle_name_valid(self.bottle_name):
            if self.bottle_name and self.bottle_name in bottlecollection.sharedCollection().bottleList():
                msg = _("There is already a bottle named '%s'. Select an existing bottle from the list or choose a different bottle name.") % self.bottle_name
                self.xml.get_object('ErrorText').display_html_safe('<body><p><img src="error"/> %s</p></body>' % cxutils.html_escape(msg))
            elif self.bottle_name:
                msg = _("Invalid bottle name '%s'. Select an existing bottle from the list or choose a different bottle name.") % self.bottle_name
                self.xml.get_object('ErrorText').display_html_safe('<body><p><img src="error"/> %s</p></body>' % cxutils.html_escape(msg))
            else:
                msg = _("You must enter a name for the new bottle.")
                self.xml.get_object('ErrorText').display_html_safe('<body><p><img src="error"/> %s</p></body>' % cxutils.html_escape(msg))
        elif self.target_bottle and self.target_bottle.analyzed() and not self.target_bottle.compatible:
            if self.target_bottle.bottlename:
                msg = _('You have selected a bottle which is incompatible with this application. It is strongly recommended to install in a new or compatible bottle.')
            else:
                msg = _('You have chosen to create a new bottle of a type which is incompatible with this application. It is strongly recommended to install in a compatible bottle.')
            self.xml.get_object('ErrorText').display_html_safe('<body><p><img src="warning"/> %s</p></body>' % cxutils.html_escape(msg))
        else:
            self.xml.get_object('ErrorText').get_buffer().set_text('')

        self.xml.get_object('DoneButton').set_sensitive(self.target_bottle is not None and (
            not self.target_bottle.is_template or self.install_task.is_new_bottle_name_valid(self.bottle_name)))

    def update_bottle_list(self):
        self.bottle_list_model.clear()

        bottles_by_category = {}
        for target_bottle in self.install_task.bottles.values():
            category = target_bottle.get_category()
            if category in bottles_by_category:
                bottles_by_category[category].append(target_bottle)
            else:
                bottles_by_category[category] = [target_bottle]

        self.bottle_list_model.append((None, _('New Bottle…'), (-1, ), True))

        for category, category_bottles in bottles_by_category.items():
            self.bottle_list_model.append((None, self.bottle_category_markup[category],
                                           (self.category_sort[category], ), False))
            for target_bottle in category_bottles:
                name = target_bottle.bottlename
                markup = '  ' + cxutils.html_escape(target_bottle.bottlename)
                sortkey = (self.category_sort[category], target_bottle.bottlename)

                self.bottle_list_model.append((name, markup, sortkey, True))

        self.bottle_list_model.foreach(self.select_bottle, None)

    def update_template_list(self):
        self.template_list_model.clear()

        bottles_by_category = {}
        for target_bottle in self.install_task.templates.values():
            category = target_bottle.get_category()
            if category in bottles_by_category:
                bottles_by_category[category].append(target_bottle)
            else:
                bottles_by_category[category] = [target_bottle]

        for category, category_bottles in bottles_by_category.items():
            self.template_list_model.append((None, self.template_category_markup[category],
                                             (self.category_sort[category], ), False))
            for target_bottle in category_bottles:
                name = target_bottle.template
                markup = '  ' + bottlemanagement.get_template_name(target_bottle.template)
                sortkey = (self.category_sort[category], bottlemanagement.get_template_key(target_bottle.template))

                self.template_list_model.append((name, markup, sortkey, True))

        self.template_list_model.foreach(self.select_template, None)

    def update_bottles(self):
        self.updating_bottles = True
        try:
            if self.install_task.profile:
                bottles = self.install_task.bottles.values()
                bottles_completed = 0
                for target_bottle in bottles:
                    if target_bottle.has_category():
                        bottles_completed += 1

                if bottles_completed != len(bottles):
                    progressbar = self.xml.get_object('Progress')
                    progressbar.set_fraction(float(bottles_completed) / len(bottles))
                    self.xml.get_object('Progress').show()
                    self.xml.get_object('BottleWidgets').hide()
                    return

            self.update_bottle_list()
            self.update_template_list()

            self.xml.get_object('Progress').hide()
            self.xml.get_object('BottleWidgets').show()
        finally:
            self.updating_bottles = False

    def select_bottle(self, model, _path, treeiter, _userdata):
        if not self.target_bottle:
            self.xml.get_object('BottleComboBox').set_active_iter(None)
            return True

        name = None
        if not self.target_bottle.is_template:
            name = self.target_bottle.bottlename

        if model.get_value(treeiter, 0) == name:
            self.xml.get_object('BottleComboBox').set_active_iter(treeiter)
            return True # stop iteration

        return False # continue iteration

    def select_template(self, model, _path, treeiter, _userdata):
        template = None
        if self.target_bottle and self.target_bottle.is_template:
            template = self.target_bottle.template
        else:
            template = self.install_task.new_bottle_template

        if model.get_value(treeiter, 0) == template:
            self.xml.get_object('BottleTypeComboBox').set_active_iter(treeiter)
            return True # stop iteration

        return False # continue iteration

    @staticmethod
    def _bottle_list_cmp(treemodel, iter1, iter2, _userdata):
        key1 = treemodel.get(iter1, 2)
        key2 = treemodel.get(iter2, 2)
        return cxutils.cmp(key1, key2)

    def on_BottleComboBox_changed(self, widget):
        if self.updating_bottles:
            return

        iterator = widget.get_active_iter()
        selected_bottle_name = widget.get_model().get_value(iterator, 0)
        if not selected_bottle_name:
            # We're creating a new bottle
            self.template_list_model.foreach(self.select_template, None)
            self.on_BottleTypeComboBox_changed()
            return

        self.target_bottle = self.install_task.bottles.get(selected_bottle_name)

        self.update_new_bottle_widgets()
        self.update_bottle_warnings()

    def on_BottleTypeComboBox_changed(self, widget=None):
        if self.updating_bottles:
            return

        if not widget:
            widget = self.xml.get_object('BottleTypeComboBox')

        iterator = widget.get_active_iter()
        selected_template = widget.get_model().get_value(iterator, 0)

        self.target_bottle = self.install_task.templates.get(selected_template)

        self.update_new_bottle_widgets()
        self.update_bottle_warnings()

    def on_CancelButton_clicked(self, _widget):
        self.destroy()

    def on_DoneButton_clicked(self, _widget):
        template = None
        if self.target_bottle and self.target_bottle.is_template:
            template = self.target_bottle.template

        if template is None:
            self.install_task.target_bottle = self.target_bottle
        else:
            self.install_task.set_create_new_bottle(self.bottle_name, template)

        self.xml.get_object('SelectBottleDialog').close()

    def on_NewBottleNameEntry_changed(self, widget):
        self.bottle_name = cxutils.string_to_unicode(widget.get_text())
        self.update_bottle_warnings()

    @staticmethod
    def on_NewBottleNameEntry_delete_text(caller, start, stop):
        name = caller.get_text()
        name = name[:start] + name[stop:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name('delete-text')

    @staticmethod
    def on_NewBottleNameEntry_insert_text(caller, new_text, _length, _user_data):
        name = caller.get_text()
        position = caller.get_position()
        name = name[:position] + new_text + name[position:]
        if not cxutils.is_valid_bottlename(name):
            caller.stop_emission_by_name('insert-text')

    # InstallTask delegate functions
    def profileChanged(self):
        self.update_bottles()
        self.update_bottle_warnings()

    def categorizedBottle_(self, _target_bottle):
        self.update_bottles()

    def analyzedBottle_(self, target_bottle):
        if target_bottle is self.target_bottle:
            self.update_bottle_warnings()

    def bottleCreateChanged(self):
        self.update_bottle_warnings()

    def bottleNewnameChanged(self):
        self.update_bottle_warnings()

    def bottleTemplateChanged(self):
        self.bottle_list_model.foreach(self.select_bottle, None)
        self.template_list_model.foreach(self.select_template, None)
        self.update_bottle_warnings()

    def bottleNameChanged(self):
        self.bottle_list_model.foreach(self.select_bottle, None)
        self.template_list_model.foreach(self.select_template, None)
        self.update_bottle_warnings()

    template_category_markup = {
        installtask.CAT_NONE: '<b>' + _('Bottle Types') + '</b>',
        installtask.CAT_RECOMMENDED: '<b>' + _('Recommended Bottle Types') + '</b>',
        installtask.CAT_COMPATIBLE: '<b>' + _('Compatible Bottle Types') + '</b>',
        installtask.CAT_INCOMPATIBLE: '<b>' + _('Incompatible Bottle Types') + '</b>'
    }

    bottle_category_markup = {
        installtask.CAT_NONE: '<b>' + _('Bottles') + '</b>',
        installtask.CAT_RECOMMENDED: '<b>' + _('Recommended Bottles') + '</b>',
        installtask.CAT_COMPATIBLE: '<b>' + _('Compatible Bottles') + '</b>',
        installtask.CAT_INCOMPATIBLE: '<b>' + _('Incompatible Bottles') + '</b>'
    }

    category_sort = {
        installtask.CAT_NONE: 0,
        installtask.CAT_RECOMMENDED: 1,
        installtask.CAT_COMPATIBLE: 2,
        installtask.CAT_INCOMPATIBLE: 3,
    }
