﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/wafv2/WAFV2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace WAFV2 {
namespace Model {

/**
 * <p>High-level information for a managed rule set. </p>  <p>This is
 * intended for use only by vendors of managed rule sets. Vendors are Amazon Web
 * Services and Amazon Web Services Marketplace sellers. </p> <p>Vendors, you can
 * use the managed rule set APIs to provide controlled rollout of your versioned
 * managed rule group offerings for your customers. The APIs are
 * <code>ListManagedRuleSets</code>, <code>GetManagedRuleSet</code>,
 * <code>PutManagedRuleSetVersions</code>, and
 * <code>UpdateManagedRuleSetVersionExpiryDate</code>.</p> <p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/wafv2-2019-07-29/ManagedRuleSetSummary">AWS
 * API Reference</a></p>
 */
class ManagedRuleSetSummary {
 public:
  AWS_WAFV2_API ManagedRuleSetSummary() = default;
  AWS_WAFV2_API ManagedRuleSetSummary(Aws::Utils::Json::JsonView jsonValue);
  AWS_WAFV2_API ManagedRuleSetSummary& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_WAFV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the managed rule set. You use this, along with the rule set ID,
   * to identify the rule set.</p> <p>This name is assigned to the corresponding
   * managed rule group, which your customers can access and use. </p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ManagedRuleSetSummary& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the managed rule set. The ID is returned in the
   * responses to commands like <code>list</code>. You provide it to operations like
   * <code>get</code> and <code>update</code>.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  ManagedRuleSetSummary& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the set that helps with identification. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ManagedRuleSetSummary& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token used for optimistic locking. WAF returns a token to your
   * <code>get</code> and <code>list</code> requests, to mark the state of the entity
   * at the time of the request. To make changes to the entity associated with the
   * token, you provide the token to operations like <code>update</code> and
   * <code>delete</code>. WAF uses the token to ensure that no changes have been made
   * to the entity since you last retrieved it. If a change has been made, the update
   * fails with a <code>WAFOptimisticLockException</code>. If this happens, perform
   * another <code>get</code>, and use the new token returned by that operation. </p>
   */
  inline const Aws::String& GetLockToken() const { return m_lockToken; }
  inline bool LockTokenHasBeenSet() const { return m_lockTokenHasBeenSet; }
  template <typename LockTokenT = Aws::String>
  void SetLockToken(LockTokenT&& value) {
    m_lockTokenHasBeenSet = true;
    m_lockToken = std::forward<LockTokenT>(value);
  }
  template <typename LockTokenT = Aws::String>
  ManagedRuleSetSummary& WithLockToken(LockTokenT&& value) {
    SetLockToken(std::forward<LockTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the entity.</p>
   */
  inline const Aws::String& GetARN() const { return m_aRN; }
  inline bool ARNHasBeenSet() const { return m_aRNHasBeenSet; }
  template <typename ARNT = Aws::String>
  void SetARN(ARNT&& value) {
    m_aRNHasBeenSet = true;
    m_aRN = std::forward<ARNT>(value);
  }
  template <typename ARNT = Aws::String>
  ManagedRuleSetSummary& WithARN(ARNT&& value) {
    SetARN(std::forward<ARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The label namespace prefix for the managed rule groups that are offered to
   * customers from this managed rule set. All labels that are added by rules in the
   * managed rule group have this prefix. </p> <ul> <li> <p>The syntax for the label
   * namespace prefix for a managed rule group is the following: </p> <p>
   * <code>awswaf:managed:&lt;vendor&gt;:&lt;rule group name&gt;</code>:</p> </li>
   * <li> <p>When a rule with a label matches a web request, WAF adds the fully
   * qualified label to the request. A fully qualified label is made up of the label
   * namespace from the rule group or web ACL where the rule is defined and the label
   * from the rule, separated by a colon: </p> <p> <code>&lt;label
   * namespace&gt;:&lt;label from rule&gt;</code> </p> </li> </ul>
   */
  inline const Aws::String& GetLabelNamespace() const { return m_labelNamespace; }
  inline bool LabelNamespaceHasBeenSet() const { return m_labelNamespaceHasBeenSet; }
  template <typename LabelNamespaceT = Aws::String>
  void SetLabelNamespace(LabelNamespaceT&& value) {
    m_labelNamespaceHasBeenSet = true;
    m_labelNamespace = std::forward<LabelNamespaceT>(value);
  }
  template <typename LabelNamespaceT = Aws::String>
  ManagedRuleSetSummary& WithLabelNamespace(LabelNamespaceT&& value) {
    SetLabelNamespace(std::forward<LabelNamespaceT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_id;
  bool m_idHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_lockToken;
  bool m_lockTokenHasBeenSet = false;

  Aws::String m_aRN;
  bool m_aRNHasBeenSet = false;

  Aws::String m_labelNamespace;
  bool m_labelNamespaceHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAFV2
}  // namespace Aws
