﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dax/DAXRequest.h>
#include <aws/dax/DAX_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DAX {
namespace Model {

/**
 */
class IncreaseReplicationFactorRequest : public DAXRequest {
 public:
  AWS_DAX_API IncreaseReplicationFactorRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "IncreaseReplicationFactor"; }

  AWS_DAX_API Aws::String SerializePayload() const override;

  AWS_DAX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the DAX cluster that will receive additional nodes.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  IncreaseReplicationFactorRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new number of nodes for the DAX cluster.</p>
   */
  inline int GetNewReplicationFactor() const { return m_newReplicationFactor; }
  inline bool NewReplicationFactorHasBeenSet() const { return m_newReplicationFactorHasBeenSet; }
  inline void SetNewReplicationFactor(int value) {
    m_newReplicationFactorHasBeenSet = true;
    m_newReplicationFactor = value;
  }
  inline IncreaseReplicationFactorRequest& WithNewReplicationFactor(int value) {
    SetNewReplicationFactor(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zones (AZs) in which the cluster nodes will be created. All
   * nodes belonging to the cluster are placed in these Availability Zones. Use this
   * parameter if you want to distribute the nodes across multiple AZs.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZones() const { return m_availabilityZones; }
  inline bool AvailabilityZonesHasBeenSet() const { return m_availabilityZonesHasBeenSet; }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  void SetAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones = std::forward<AvailabilityZonesT>(value);
  }
  template <typename AvailabilityZonesT = Aws::Vector<Aws::String>>
  IncreaseReplicationFactorRequest& WithAvailabilityZones(AvailabilityZonesT&& value) {
    SetAvailabilityZones(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  template <typename AvailabilityZonesT = Aws::String>
  IncreaseReplicationFactorRequest& AddAvailabilityZones(AvailabilityZonesT&& value) {
    m_availabilityZonesHasBeenSet = true;
    m_availabilityZones.emplace_back(std::forward<AvailabilityZonesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;
  bool m_clusterNameHasBeenSet = false;

  int m_newReplicationFactor{0};
  bool m_newReplicationFactorHasBeenSet = false;

  Aws::Vector<Aws::String> m_availabilityZones;
  bool m_availabilityZonesHasBeenSet = false;
};

}  // namespace Model
}  // namespace DAX
}  // namespace Aws
